function signal = toneBurst(sample_freq, signal_freq, signal_length, signal_offset, num_cycles, varargin)
%TONEBURST  Create an enveloped single frequency tone burst
%
% DESCRIPTION:
%       toneBurst creates an enveloped single frequency tone burst for use
%       in ultrasound simulations. If an array is given for signal_offset,
%       a matrix of tone bursts is created where each row corresponds to a
%       tone burst for particular value of the signal_offset.
%
% USAGE:
%       signal = toneBurst(sample_freq, signal_freq, signal_length, signal_offset, num_cycles,...)
%
% INPUTS:
%       sample_freq     - sampling frequency [Hz]
%       signal_freq     - frequency of the tone burst signal [Hz]
%       signal_length   - signal length in number of samples
%       signal_offset   - signal offset before the tone burst in number of
%                         samples
%       num_cycles      - number of sinusoidal oscillations
%
% OPTIONAL INPUTS:
%       Optional 'string', value pairs that may be used to modify the
%       default computational settings.
%
%       'Envelope'  - envelope used to taper the tone burst, can be set to 
%                     either 'Gaussian' (default) or 'Rectangular'
%       'Plot'      - boolean controlling whether the created tone burst
%                     is plotted
%
% OUTPUTS:
%       signal      - created tone burst
%
% USES:
%       gaussian.m
%
% ABOUT:
%       author      - Bradley Treeby
%       date        - 4th December 2009
%       last update - 19th January 2010
%       
% This function is part of the k-Wave Toolbox (http://www.k-wave.org)
% Copyright (C) 2009, 2010 Bradley Treeby and Ben Cox

% This file is part of k-Wave. k-Wave is free software: you can
% redistribute it and/or modify it under the terms of the GNU Lesser
% General Public License as published by the Free Software Foundation,
% either version 3 of the License, or (at your option) any later version.
% 
% k-Wave is distributed in the hope that it will be useful, but WITHOUT ANY
% WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
% FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for
% more details. 
% 
% You should have received a copy of the GNU Lesser General Public License
% along with k-Wave. If not, see <http://www.gnu.org/licenses/>.

% set usage defaults
num_req_input_variables = 5;
envelope = 'Gaussian';
plot_signal = false;

% replace with user defined values if provided
if nargin < num_req_input_variables
    error('Incorrect number of inputs');
elseif ~isempty(varargin)
    for input_index = 1:2:length(varargin)
        switch varargin{input_index}
            case 'Envelope'
                envelope = varargin{input_index + 1};
            case 'Plot'
                plot_signal = varargin{input_index + 1};                    
            otherwise
                error('Unknown optional input');
        end
    end
end

% calculate the temporal spacing
dt = 1/sample_freq;

% create the tone burst
tone_length = num_cycles/(signal_freq);
tone_t = 0:dt:tone_length;
tone_burst = sin(2*pi*signal_freq*tone_t);
tone_index = round(signal_offset);

% create the envelope
switch envelope
    case 'Gaussian'
        x_lim = 5;
        window_x = -x_lim:2*x_lim/(length(tone_burst)-1):x_lim;
        window = gaussian(window_x, 0, 1, 1);
    case 'Rectangular'
        window = ones(size(tone_burst));
    otherwise
        error(['Unknown envelope ' envelope]);
end

% apply the envelope
tone_burst = tone_burst.*window;

% create the signal with the offset tone burst
signal = zeros(length(tone_index), signal_length);
for offset = 1:length(tone_index)
    signal(offset, tone_index(offset) + 1:tone_index(offset) + length(tone_burst)) = tone_burst;
end

% plot the signal if required
if plot_signal
    figure;
    time_axis = (0:length(signal)-1)*dt;
    [t_sc scale prefix] = scaleSI(max(time_axis(:))); 
    plot(time_axis*scale, signal, 'k-');
    xlabel(['Time [' prefix 's]']);
    ylabel('Signal Amplitude');
end